<?php
/**
 * Pei Payment Gateway – “classic” gateway class
 *
 * This file lives in:  includes/class‑wc‑gateway‑pei.php
 *
 * It contains the original off‑site redirect flow, now namespaced so it can be
 * loaded by the new bootstrap while remaining 100 % backward‑compatible with
 * classic checkout templates.
 */

namespace Premis\Pei;

use WC_Order;
use WC_Payment_Gateway;
use WC_Product;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // No direct access.
}

class WC_Gateway_Pei extends WC_Payment_Gateway {

	/**
	 * Constructor – sets up gateway and hooks.
	 */
	public function __construct() {

		/* ── Identification ─────────────────────────────────────────────── */
		$this->id                 = 'dg_pei';
		$this->method_title       = 'Pei';
		$this->method_description = __( 'Borga með Pei', 'pei' );
		$this->has_fields         = true;
		$this->icon               = plugins_url(
			'/assets/pei_logo_small.png',
			PEI_WC_PLUGIN_DIR . '/pei-woocommerce.php'
		);

		/* ── Load settings ──────────────────────────────────────────────── */
		$this->init_form_fields();
		$this->init_settings();

		$this->title            = $this->get_option( 'title' );
		$this->merchantId       = $this->get_option( 'merchantId' );
		$this->clientId         = $this->get_option( 'user' );
		$this->secret           = $this->get_option( 'secret' );
		$this->description      = $this->get_option( 'description' );
		$this->test             = $this->get_option( 'test' );
		$this->logging          = $this->get_option( 'logging' );

		/* ── Pei endpoints (prod / stage) ──────────────────────────────── */
		$this->auth_url    = $this->test === 'yes'
			? 'https://authstaging.pei.is/core/connect/token'
			: 'https://auth.pei.is/core/connect/token';

		$this->order_url   = $this->test === 'yes'
			? 'https://externalapistaging.pei.is/api/orders'
			: 'https://api.pei.is/api/orders';

		$this->payment_url = $this->test === 'yes'
			? 'https://gattinstaging.pei.is/'
			: 'https://gattin.pei.is/';

		/* ── Hooks ──────────────────────────────────────────────────────── */
		add_action(
			'woocommerce_api_wc_pei_gateway',
			[ $this, 'handle_callback' ]
		);

		add_action(
			'woocommerce_update_options_payment_gateways_' . $this->id,
			[ $this, 'process_admin_options' ]
		);
	}

	/* --------------------------------------------------------------------- *
	 * Settings UI
	 * --------------------------------------------------------------------- */

	public function init_form_fields() {
		$this->form_fields = [
			'enabled'     => [
				'title'   => __( 'Enable / Disable', 'pei' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable Pei', 'pei' ),
				'default' => 'yes',
			],
			'title'       => [
				'title'       => __( 'Title', 'pei' ),
				'type'        => 'text',
				'description' => __( 'Shown to the customer during checkout.', 'pei' ),
				'default'     => __( 'Pei', 'pei' ),
				'desc_tip'    => true,
			],
			'description' => [
				'title' => __( 'Customer Message', 'pei' ),
				'type'  => 'textarea',
			],
			'user'        => [
				'title'       => __( 'Pei User', 'pei' ),
				'type'        => 'text',
				'default'     => 'democlient',
				'description' => __( 'API client ID', 'pei' ),
			],
			'secret'      => [
				'title'       => __( 'Pei Secret', 'pei' ),
				'type'        => 'textarea',
				'default'     => 'demosecret',
				'description' => __( 'API secret (see your Pei account)', 'pei' ),
			],
			'merchantId'  => [
				'title'       => __( 'Merchant ID', 'pei' ),
				'type'        => 'text',
				'default'     => '4',
				'description' => __( 'Provided by Pei', 'pei' ),
			],
			'test'        => [
				'title'   => __( 'Test mode', 'pei' ),
				'type'    => 'checkbox',
				'label'   => __( 'Use Pei staging environment', 'pei' ),
				'default' => 'yes',
			],
		];
	}

	/* --------------------------------------------------------------------- *
	 * Checkout fields & processing
	 * --------------------------------------------------------------------- */

	public function payment_fields() {
		if ( $this->description ) {
			echo wpautop( wptexturize( $this->description ) );
		}
		echo '<input type="text" name="dg_pei-ssn" placeholder="Kennitala" />';
	}

	public function process_payment( $order_id ) {

		$order = wc_get_order( $order_id );
		$token = $this->authorize();

		if ( ! $token ) {
			$order->add_order_note( __( 'Could not obtain token from Pei', 'pei' ) );
			return [
				'result'   => 'failed',
				'redirect' => wc_get_checkout_url(),
			];
		}

		return $this->create_order_at_pei( $token, $order );
	}

	/* --------------------------------------------------------------------- *
	 * Pei API helpers
	 * --------------------------------------------------------------------- */

	private function authorize() {

		$payload = [
			'grant_type' => 'client_credentials',
			'scope'      => 'externalapi',
		];

		$auth_header = 'Basic ' . base64_encode(
			$this->test === 'yes'
				? 'democlient:demosecret'
				: "{$this->clientId}:{$this->secret}"
		);

		$response = wp_remote_post(
			$this->auth_url,
			[
				'headers' => [ 'Authorization' => $auth_header ],
				'body'    => http_build_query( $payload ),
			]
		);

		if ( is_wp_error( $response ) || empty( $response['body'] ) ) {
			return false;
		}

		$body = json_decode( $response['body'] );
		return "{$body->token_type} {$body->access_token}";
	}

	private function create_order_at_pei( $token, WC_Order $order ) {

		$postback = add_query_arg(
			[
				'wc-api'   => 'wc_pei_gateway',
				'order_id' => $order->get_id(),
			],
			home_url( '/' )
		);

		$payload = [
			'merchantId'       => $this->test === 'yes' ? 4 : $this->merchantId,
			'amount'           => $order->get_total(),
			'successReturnUrl' => $this->get_return_url( $order ),
			'cancelReturnUrl'  => wc_get_checkout_url(),
			'postbackUrl'      => $postback,
			'reference'        => $order->get_id(),
			'buyer'            => [
				'name'         => "{$order->get_billing_first_name()} {$order->get_billing_last_name()}",
				'email'        => $order->get_billing_email(),
				'mobileNumber' => $order->get_billing_phone(),
				'ssn'          => str_replace( [ ' ', '-' ], '', $_POST['dg_pei-ssn'] ?? '' ),
			],
		];

		foreach ( $order->get_items() as $item ) {
			$product    = wc_get_product( $item->get_product_id() );
			$unit_price = $product->get_price();
			$qty        = $item->get_quantity();

			$payload['items'][] = [
				'code'      => $product->get_sku(),
				'name'      => $item->get_name(),
				'quantity'  => $qty,
				'amount'    => $unit_price * $qty,
				'unitPrice' => $unit_price,
			];
		}

		$response = wp_remote_post(
			$this->order_url,
			[
				'headers' => [ 'Authorization' => $token ],
				'body'    => $payload,
			]
		);

		if ( is_wp_error( $response ) || empty( $response['body'] ) ) {
			$order->add_order_note( __( 'Pei rejected the order payload', 'pei' ) );
			return [
				'result'   => 'failed',
				'redirect' => wc_get_checkout_url(),
			];
		}

		$body = json_decode( $response['body'] );
		$order->update_meta_data( 'pei_order_id', $body->orderId );
		$order->add_order_note( __( 'Redirecting customer to Pei', 'pei' ) );
		$order->save();

		return [
			'result'   => 'success',
			'redirect' => $this->payment_url . $body->orderId,
		];
	}

	/* --------------------------------------------------------------------- *
	 * Callback handler
	 * --------------------------------------------------------------------- */

	public function handle_callback() {

		$data = json_decode( file_get_contents( 'php://input' ), true );

		if ( ! $data ) { // fallback if host strips JSON
			$order_id      = intval( $_GET['order_id'] ?? 0 );
			$order         = wc_get_order( $order_id );
			$data['orderId'] = $order ? $order->get_meta( 'pei_order_id' ) : '';
		}

		$token    = $this->authorize();
		$response = wp_remote_get(
			$this->order_url . '/' . $data['orderId'],
			[
				'headers' => [ 'Authorization' => $token ],
			]
		);

		if ( is_wp_error( $response ) ) {
			status_header( 400 );
			exit;
		}

		$body = json_decode( $response['body'] );
		if ( $body->isConfirmed ) {
			$order = wc_get_order( $body->reference );
			$order->payment_complete();
			$order->add_order_note( __( 'Payment completed via Pei', 'pei' ) );
		}

		status_header( 200 );
		exit;
	}
}
